<?php

namespace App\Services;

use App\Models\Payment;
use Exception;
use Illuminate\Support\Facades\Log;
use MercadoPago\Client\Payment\PaymentClient;
use MercadoPago\MercadoPagoConfig;
use MercadoPago\Exceptions\MPApiException;

class MercadoPagoService
{
    protected $client;

    public function __construct()
    {
        MercadoPagoConfig::setAccessToken(config('services.mercadopago.access_token'));
        $this->client = new PaymentClient();
    }

    public function createPayment(array $paymentData)
    {
        try {
            $payment = $this->client->create([
                'transaction_amount' => $paymentData['amount'],
                'description' => $paymentData['description'],
                'payment_method_id' => $paymentData['payment_method'],
                'payer' => [
                    'email' => $paymentData['payer_email'],
                    'first_name' => $paymentData['payer_first_name'],
                    'last_name' => $paymentData['payer_last_name'],
                    'identification' => [
                        'type' => 'CPF',
                        'number' => $paymentData['payer_document']
                    ]
                ],
                'notification_url' => route('webhook.mercadopago'),
            ]);

            return [
                'success' => true,
                'data' => $payment
            ];
        } catch (MPApiException $e) {
            Log::error('Mercado Pago API Error: ' . $e->getMessage());
            return [
                'success' => false,
                'error' => $e->getMessage()
            ];
        } catch (Exception $e) {
            Log::error('Mercado Pago Error: ' . $e->getMessage());
            return [
                'success' => false,
                'error' => $e->getMessage()
            ];
        }
    }

    public function createPixPayment(array $paymentData)
    {
        $paymentData['payment_method'] = 'pix';
        return $this->createPayment($paymentData);
    }

    public function createCreditCardPayment(array $paymentData)
    {
        $paymentData['payment_method'] = 'credit_card';
        
        try {
            $payment = $this->client->create([
                'transaction_amount' => $paymentData['amount'],
                'token' => $paymentData['token'],
                'description' => $paymentData['description'],
                'installments' => $paymentData['installments'] ?? 1,
                'payment_method_id' => $paymentData['card_type'],
                'payer' => [
                    'email' => $paymentData['payer_email'],
                    'identification' => [
                        'type' => 'CPF',
                        'number' => $paymentData['payer_document']
                    ]
                ],
                'notification_url' => route('webhook.mercadopago'),
            ]);

            return [
                'success' => true,
                'data' => $payment
            ];
        } catch (MPApiException $e) {
            Log::error('Mercado Pago API Error: ' . $e->getMessage());
            return [
                'success' => false,
                'error' => $e->getMessage()
            ];
        } catch (Exception $e) {
            Log::error('Mercado Pago Error: ' . $e->getMessage());
            return [
                'success' => false,
                'error' => $e->getMessage()
            ];
        }
    }

    public function getPaymentStatus($paymentId)
    {
        try {
            $payment = $this->client->get($paymentId);
            return [
                'success' => true,
                'data' => $payment
            ];
        } catch (MPApiException $e) {
            Log::error('Mercado Pago API Error: ' . $e->getMessage());
            return [
                'success' => false,
                'error' => $e->getMessage()
            ];
        } catch (Exception $e) {
            Log::error('Mercado Pago Error: ' . $e->getMessage());
            return [
                'success' => false,
                'error' => $e->getMessage()
            ];
        }
    }

    public function processWebhook($data)
    {
        try {
            if ($data['type'] === 'payment') {
                $paymentId = $data['data']['id'];
                $paymentInfo = $this->getPaymentStatus($paymentId);
                
                if ($paymentInfo['success']) {
                    $payment = Payment::where('payment_id', $paymentId)->first();
                    
                    if ($payment) {
                        $status = $paymentInfo['data']['status'];
                        $payment->status = $status;
                        $payment->payment_data = $paymentInfo['data'];
                        
                        if ($status === 'approved') {
                            $payment->paid_at = now();
                        }
                        
                        $payment->save();
                        
                        return [
                            'success' => true,
                            'message' => 'Payment updated successfully'
                        ];
                    }
                }
            }
            
            return [
                'success' => false,
                'message' => 'Payment not found or invalid webhook data'
            ];
        } catch (Exception $e) {
            Log::error('Webhook processing error: ' . $e->getMessage());
            return [
                'success' => false,
                'error' => $e->getMessage()
            ];
        }
    }
}