<?php

namespace App\Filament\Resources\EventResource\RelationManagers;

use Filament\Forms;
use Filament\Tables;
use App\Models\Round;
use Filament\Forms\Get;
use App\Models\Modality;
use Filament\Forms\Form;
use Filament\Tables\Table;
use App\Models\Inscription;
use Illuminate\Support\Facades\DB;
use Illuminate\Database\Eloquent\Model;
use Filament\Notifications\Notification;
use Illuminate\Database\Eloquent\Builder;
use Filament\Resources\RelationManagers\RelationManager;

class InscriptionsRelationManager extends RelationManager
{
    protected static string $relationship = 'inscriptions';



    protected static ?string $title = 'Inscrições';

    public $currentModality = null;


    public static function getModelLabel(): string
    {
        return __('Inscription');
    }



    public function form(Form $form): Form
    {
        return $form
            ->schema([
                Forms\Components\Section::make([
                    Forms\Components\Select::make('event_id')
                        ->label('Evento')
                        ->prefixIcon('heroicon-o-calendar-days')
                        ->relationship('event', 'name')
                        ->default($this->ownerRecord->id)
                        ->disabled()
                        ->dehydrated(),
                    Forms\Components\Select::make('modality_id')
                        ->label('Modalidades')
                        ->prefixIcon('heroicon-o-clipboard')
                        ->relationship('modality', 'name', fn($query) =>
                        $query->where('status_modality', 0)
                            ->whereHas('events', fn($query) =>
                            $query->where('events.id', $this->getOwnerRecord()->id)
                            )
                        )
                        ->live()
                        ->reactive()
                        ->afterStateUpdated(function ($state, callable $set) {
                            $this->currentModality = $state;
                            $set('inscription_value', \App\Models\Modality::find($state)->inscription_value ?? '');

                            // Resetar o campo de participantes quando mudar a modalidade
                            $set('participant_id', []);
                        })
                        ->default($this->currentModality)
                        ->required(),
                    Forms\Components\Select::make('type_payment')
                        ->label('Tipo de pagamento')
                        ->prefixIcon('heroicon-o-currency-dollar')
                        ->options([
                            'PIX' => 'PIX',
                            'CREDITO' => 'CRÉDITO',
                            'CORTESIA' => 'CORTESIA',
                            'DINHEIRO' => 'DINHEIRO',
                            'DEBITO' => 'DÉBITO',
                        ]),

                    Forms\Components\TextInput::make('name')
                        ->label('Nome do grupo')
                        ->visible(fn (Get $get) =>
                            $get('modality_id') && Modality::find($get('modality_id'))?->rules?->participant_rule !== 'Individual'
                        ),
                    $this->createParticipantSelect('Individual'),
                    $this->createParticipantSelect('Dupla', 2),
                    $this->createParticipantSelect('Trio', 3),
                    $this->createParticipantSelect('Quarteto', 4),
                    $this->createParticipantSelect('Quinteto', 5),
                ])
            ]);
    }

    public function table(Table $table): Table
    {
        return $table
            ->recordTitleAttribute('name')
            ->columns([
                Tables\Columns\TextColumn::make('modality.name')
                    ->label('Modalidade')
                    ->sortable(),
                Tables\Columns\TextColumn::make('participants.name')
                    ->label('Participante(s)')
                    ->listWithLineBreaks(),
                Tables\Columns\TextColumn::make('payment_method')
                    ->label('Forma de pagamento')
                    ->listWithLineBreaks(),
                Tables\Columns\TextColumn::make('created_at')
                    ->label('Data da inscrição')
                    ->dateTime('d/m/Y'),
                Tables\Columns\ToggleColumn::make('inscription_status')
                    ->label('Ativar/Desativar inscrição')
                    ->onColor('success')
                    ->offColor('danger')
            ])
            ->defaultSort('inscription_status', 'DESC')
            ->filters([
                //
            ])
            ->headerActions([
                Tables\Actions\CreateAction::make()
                    ->before(function (Tables\Actions\CreateAction $action, array $data) {
                        $modalityId = $data['modality_id'] ?? null;
                        $participantId = $data['participant_id_individual'] ?? null;
                        $eventId = $this->getOwnerRecord()->id ?? null;
                        $modality = Modality::find($modalityId);

                        if ($modality) {
                            if ($modality->rules->active_reinscription_lap) {
                                $exceedsRoundInscriptionLimit = Round::where('modality_id', $modality->id)
                                    ->where('event_id', $eventId)
                                    ->where('number', $modality->rules->reinscription_lap)
                                    ->where('round_status', 'finish')
                                    ->exists();

                                if ($exceedsRoundInscriptionLimit) {
                                    Notification::make()
                                        ->title('Operação não permitida!')
                                        ->danger()
                                        ->body('Inscrição pode ser feita até a volta ' . $modality->rules->reinscription_lap . '.')
                                        ->send();
                                    $action->halt();
                                }
                            }else {
                                $roundStarted = Round::where('modality_id', $modality->id)
                                    ->where('event_id', $eventId)
                                    ->where('number', 1)
                                    ->where('round_status', 'finish')
                                    ->exists();

                                if ($roundStarted) {
                                    Notification::make()
                                        ->title('Operação não permitida!')
                                        ->danger()
                                        ->body('Modalidade em andamento, não é permitido novas inscrições.')
                                        ->send();
                                    $action->halt();
                                }
                            }

                            $participantAlreadyInscription = Round::where('modality_id', $modality->id)
                                ->where('event_id', $eventId)
                                ->where('participant_id', $participantId)
                                ->get();

                            if($participantAlreadyInscription->isNotEmpty()) {
                                foreach ($participantAlreadyInscription as $item) {
                                    Inscription::where('id',$item->inscription_id)
                                        ->update([
                                            'reborn' => false
                                        ]);
                                }
                            }
                        }
                    })
                    ->after(function (Tables\Actions\CreateAction $action) {
                        $inscription = $action->getRecord();

                        if ($inscription->modality->rules->participant_rule == 'Individual') {
                            $participantIds = $inscription->participants->pluck('id')->toArray();
                            $existingInscriptions = Inscription::where('modality_id', $inscription->modality_id)
                                ->whereHas('participants', function ($query) use ($participantIds, $inscription) {
                                    $query->whereIn('participant_id', $participantIds)
                                        ->where('inscription_id', '<>', $inscription->id);
                                })
                                ->first();

                            if ($existingInscriptions) {
                                $inscription->update(['reborn' => false]);
                            }
                        }

                    }),
            ])
            ->actions([
                Tables\Actions\EditAction::make()
                    ->color('success')
                    ->using(fn (Model $record, array $data) => $this->handleRecordUpdate($record, $data)),
                Tables\Actions\DeleteAction::make(),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),
                ]),
            ]);
    }

    private function createParticipantSelect(string $rule, int $maxItems = 1): Forms\Components\Select
    {
        return Forms\Components\Select::make("participant_id_" . strtolower($rule))
            ->label($maxItems > 1 ? 'Participantes' : 'Participante')
            ->visible(fn (Get $get) => Modality::find($get('modality_id'))?->rules?->participant_rule === $rule)
            ->prefixIcon($maxItems > 1 ? 'heroicon-o-user-group' : 'heroicon-o-user')
            ->relationship('participants','name',  function ($query, $get) {
                $modalityId = $get('modality_id');
                $modality = Modality::find($modalityId);
                if ($modality->rules->participant_rule <> 'Individual') {
                    return $query;
                }
                return $query->where(function ($subQuery) use ($modalityId) {
                    $subQuery->whereDoesntHave('inscriptions', function ($q) use ($modalityId) {
                        $q->where('modality_id', $modalityId);
                    })
                        ->orWhereHas('inscriptions', function ($q) use ($modalityId) {
                            $q->where('modality_id', $modalityId)
                                ->where('lifeless', true)
                                ->where('reborn', true);
                        });
                });
            })
            ->createOptionForm([
                Forms\Components\TextInput::make('name')
                    ->required()
                    ->afterStateUpdated(function ($state, $set) {
                        $set('name', strtoupper($state));
                    }),
            ])
            ->searchable(['name'])
            ->preload()
            ->required()
            ->multiple($maxItems > 1)
            ->dehydrated(false)
            ->when($maxItems > 1, fn ($component) => $component->maxItems($maxItems));
    }

    protected function handleRecordCreation(array $data): false|Model
    {
        $modality = Modality::find($data['modality_id']);

        $inscription = parent::handleRecordCreation([
            'modality_id' => $data['modality_id'],
            'event_id' => $data['event_id'],
            'name' => $data['name'] ?? null,

        ]);

        $participantIds = match ($modality->rules->participant_rule) {
            'Individual' => [$data['participant_id_individual']],
            'Dupla' => $data['participant_id_dupla'],
            'Trio' => $data['participant_id_trio'],
            'Quarteto' => $data['participant_id_quarteto'],
            'Quinteto' => $data['participant_id_quinteto'],
        };

        $inscription->participants()->attach($participantIds);
        return $inscription;
    }

    protected function handleRecordUpdate(Model $record, array $data): Model
    {
        $record->update([
            'modality_id' => $data['modality_id'],
            'event_id' => $data['event_id'],
            'name' => $data['name'] ?? null,
        ]);

        $modality = Modality::find($data['modality_id']);
        $participantIds = match ($modality->rules->participant_rule) {
            'Individual' => [$data['participant_id_individual']],
            'Dupla' => $data['participant_id_dupla'],
            'Trio' => $data['participant_id_trio'],
            'Quarteto' => $data['participant_id_quarteto'],
            'Quinteto' => $data['participant_id_quinteto'],
        };

        $record->participants()->sync($participantIds);
        return $record;
    }
}
